\name{inla.spde2.matern}
\alias{inla.spde2.matern}
\title{
  Matern SPDE model object for INLA
}
\description{
  Create an \code{inla.spde2} model object for a Matern model. Use
  \code{inla.spde2.pcmatern} instead for a PC prior for the parameters.
}
\usage{
inla.spde2.matern(mesh,
                  alpha = 2,
                  param = NULL,
                  constr = FALSE,
                  extraconstr.int = NULL,
                  extraconstr = NULL,
                  fractional.method = c("parsimonious", "null"),
                  B.tau = matrix(c(0,1,0),1,3),
                  B.kappa = matrix(c(0,0,1),1,3),
                  prior.variance.nominal = 1,
                  prior.range.nominal = NULL,
                  prior.tau = NULL,
                  prior.kappa = NULL,
                  theta.prior.mean = NULL,
                  theta.prior.prec = 0.1,
                  n.iid.group = 1,
                  ...)
}
\arguments{
  \item{mesh}{
    The mesh to build the model on, as an \code{\link{inla.mesh}} or
    \code{\link{inla.mesh.1d}} object.
  }
  \item{alpha}{Fractional operator order, \eqn{0<\alpha\leq 2} supported. (\eqn{\nu=\alpha-d/2})}
  \item{param}{Parameter, e.g. generated by \code{param2.matern.orig}}
  \item{constr}{If \code{TRUE}, apply an integrate-to-zero constraint.
    Default \code{FALSE}.}
  \item{extraconstr.int}{Field integral constraints.}
  \item{extraconstr}{Direct linear combination constraints on the basis
    weights.}
  \item{fractional.method}{Specifies the approximation method to use for
    fractional (non-integer) \code{alpha} values. \code{'parsimonious'}
    gives an overall approximate minimal covariance error, \code{'null'}
    uses approximates low-order properties.}
  \item{B.tau}{Matrix with specification of log-linear model for \eqn{\tau}.}
  \item{B.kappa}{Matrix with specification of log-linear model for \eqn{\kappa}.}
  \item{prior.variance.nominal}{Nominal prior mean for the field variance}
  \item{prior.range.nominal}{Nominal prior mean for the spatial range}
  \item{prior.tau}{Prior mean for tau (overrides \code{prior.variance.nominal})}
  \item{prior.kappa}{Prior mean for kappa (overrides \code{prior.range.nominal})}
  \item{theta.prior.mean}{(overrides \code{prior.*})}
  \item{theta.prior.prec}{Scalar, vector or matrix, specifying the joint prior
    precision for \eqn{theta}.}
  \item{n.iid.group}{If greater than 1, build an explicitly iid
    replicated model, to support constraints applied to the combined
    replicates, for example in a time-replicated spatial
    model. Constraints can either be specified for a single mesh, in
    which case it's applied to the average of the replicates
    (\code{ncol(A)} should be \code{mesh$n} for 2D meshes,
    \code{mesh$m} for 1D), or as general constraints on the collection
    of replicates (\code{ncol(A)} should be \code{mesh$n * n.iid.group}
    for 2D meshes, \code{mesh$m * n.iid.group} for 1D).}
  \item{\dots}{Additional parameters for special uses.}
}
\details{
  This method constructs a Matern SPDE model, with spatial scale
  parameter \eqn{\kappa(u)} and variance rescaling parameter \eqn{\tau(u)}.

  \deqn{(\kappa^2(u)-\Delta)^{\alpha/2}(\tau(u)
    x(u))=W(u)}{(kappa^2(u)-Delta)^(alpha/2) (tau(u) x(u)) = W(u)}

  Stationary models are supported for
  \eqn{0 < \alpha \leq 2}, with spectral approximation methods used for
  non-integer \eqn{\alpha}, with approximation method determined by
  \code{fractional.method}.
  
  Non-stationary models are supported for \eqn{\alpha=2} only, with
  \itemize{
    \item{}{
      \eqn{\log\tau(u) = B^\tau_0(u) + \sum_{k=1}^p B^\tau_k(u)
	\theta_k}{log tau(u) = B.tau_0(u) + sum_{k=1}^p B.tau_k(u) theta_k}
    }
    \item{}{
      \eqn{\log\kappa(u) = B^{\kappa}_0(u) + \sum_{k=1}^p B^{\kappa}_k(u)
	\theta_k}{log kappa(u) = B.kappa_0(u) + sum_{k=1}^p B.kappa_k(u) theta_k}
    }
  }

  The same parameterisation is used in the stationary cases, but with
  \eqn{B^\tau_0}, \eqn{B^\tau_k}, \eqn{B^\kappa_0}, and \eqn{B^\tau_k}
  constant across \eqn{u}.

  Integration and other general linear constraints are supported via the
  \code{constr}, \code{extraconstr.int}, and \code{extraconstr}
  parameters, which also interact with \code{n.iid.group}.
}
\value{
  An \code{inla.spde2} object.
%%  If it is a LIST, use
%%  \item{comp1 }{Description of 'comp1'}
%%  \item{comp2 }{Description of 'comp2'}
%% ...
}
%\references{
%}
\author{
  Finn Lindgren \email{finn.lindgren@gmail.com}
}
%\note{
%}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
  \code{\link{inla.mesh.2d}}, \code{\link{inla.mesh.create}},
  \code{\link{inla.mesh.1d}}, \code{\link{inla.mesh.basis}},
  \code{\link{inla.spde2.pcmatern}},
  \code{\link{inla.spde2.generic}}
}
\examples{
n = 100
field.fcn = function(loc) (10*cos(2*pi*2*(loc[,1]+loc[,2])))
loc = matrix(runif(n*2),n,2)
## One field, 2 observations per location
idx.y = rep(1:n,2)
y = field.fcn(loc[idx.y,]) + rnorm(length(idx.y))

mesh = inla.mesh.create(loc, refine=list(max.edge=0.05))
spde = inla.spde2.matern(mesh)
data = list(y=y, field=mesh$idx$loc[idx.y])
formula = y ~ -1 + f(field, model=spde)
result = inla(formula, data=data, family="normal")

## Plot the mesh structure:
plot(mesh)
\donttest{
if (require(rgl)) {
  col.pal = colorRampPalette(c("blue","cyan","green","yellow","red"))
  ## Plot the posterior mean:
  plot(mesh, rgl=TRUE,
       result$summary.random$field[,"mean"],
       color.palette = col.pal)
  ## Plot residual field:
  plot(mesh, rgl=TRUE,
       result$summary.random$field[,"mean"]-field.fcn(mesh$loc),
       color.palette = col.pal)
}
}
result.field = inla.spde.result(result, "field", spde)
plot(result.field$marginals.range.nominal[[1]])
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
%\keyword{ fmesher }
