% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spde.common.R
\name{inla.stack.remove.unused}
\alias{inla.stack.remove.unused}
\alias{inla.stack.compress}
\alias{inla.stack}
\alias{inla.stack.sum}
\alias{inla.stack.join}
\alias{inla.stack.index}
\alias{inla.stack.LHS}
\alias{inla.stack.RHS}
\alias{inla.stack.data}
\alias{inla.stack.A}
\title{Data stacking for advanced INLA models}
\usage{
inla.stack.remove.unused(stack)

inla.stack.compress(stack, remove.unused = TRUE)

inla.stack(..., compress = TRUE, remove.unused = TRUE)

inla.stack.sum(
  data,
  A,
  effects,
  tag = "",
  compress = TRUE,
  remove.unused = TRUE
)

inla.stack.join(..., compress = TRUE, remove.unused = TRUE)

inla.stack.index(stack, tag)

inla.stack.LHS(stack)

inla.stack.RHS(stack)

inla.stack.data(stack, ...)

inla.stack.A(stack)
}
\arguments{
\item{stack}{A \code{inla.data.stack} object, created by a call to
\code{inla.stack}, \code{inla.stack.sum}, or \code{inla.stack.join}.}

\item{remove.unused}{If \code{TRUE}, compress the model by removing rows of
effects corresponding to all-zero columns in the \code{A} matrix (and
removing those columns).}

\item{...}{For \code{inla.stack.join}, two or more data stacks of class
\code{inla.data.stack}, created by a call to \code{inla.stack},
\code{inla.stack.sum}, or \code{inla.stack.join}. For
\code{inla.stack.data}, a list of variables to be joined with the data list.}

\item{compress}{If \code{TRUE}, compress the model by removing duplicated
rows of effects, replacing the corresponding A-matrix columns with a single
column containing the sum.}

\item{data}{A list or codedata.frame of named data vectors. Scalars are
expanded to match the number of rows in the A matrices, or any non-scalar
data vectors. An error is given if the input is inconsistent.}

\item{A}{A list of observation matrices. Scalars are expanded to diagonal
matrices matching the effect vector lengths. An error is given if the input
is inconsistent or ambiguous.}

\item{effects}{A collection of effects/predictors.  Each list element
corresponds to an observation matrix, and must either be a single vector, a
list of vectors, or a \code{data.frame}. Single-element effect vectors are
expanded to vectors matching the number of columns in the corresponding A
matrix.  An error is given if the input is inconsistent or ombiguous.}

\item{tag}{A string specifying a tag for later identification.}
}
\value{
A data stack of class \code{inla.data.stack}.
Elements: \itemize{
\item\code{data} \eqn{=(y^1, \ldots, y^p, \tilde{x}^1, \ldots,
\tilde{x}^{n_k})}{=(y^1, \dots, y^p, tilde{x}^1, \dots,
tilde{x}^{n_k})}
\item\code{A} \eqn{=\tilde{A}}{=tilde{A}}
\item\code{data.names} List
of data names, length \eqn{p}
\item\code{effect.names} List of effect names,
length \eqn{n_k}
\item\code{n.data} Data length, \eqn{n_i}
\item\code{index}
List indexed by \code{tag}s, each element indexing into \eqn{i=1, \ldots,
n_i} }
}
\description{
Functions for combining data, effects and observation matrices into
\code{inla.stack} objects, and extracting information from such objects.
}
\details{
For models with a single effects collection, the outer list container for
\code{A} and \code{effects} may be omitted.

Component size definitions:
\itemize{
\item \eqn{n_l}{n_l} effect blocks
\item \eqn{n_k}{n_k} effects
\item \eqn{n_i}{n_i} data values
\item \eqn{n_{j,l}}{n_jl} effect size for block \eqn{l}{l}
\item \eqn{n_j}{n_j} \eqn{= \sum_{l=1}^{n_l} n_{j,l}}{sum_l n_jl} total
effect size
}

Input: \describe{
\item{\code{data}}{\eqn{(y^1, \ldots, y^p)}{(y1,\dots,y2)} \eqn{p}{p}
vectors, each of length \eqn{n_i}{n_i}}
\item{\code{A}}{\eqn{(A^1, \ldots, A^{n_l})}{(A1,\dots,A2)} matrices of size
\eqn{n_i \times n_{j,l}}{n_i by n_jl}}
\item{\code{effects}}{\eqn{\left((x^{1,1},\ldots,x^{n_k,1}), \ldots,
(x^{1,n_l},\ldots,x^{n_k,n_l})\right)}{((x_[1,1],\dots,x_[n_k,1]),\dots(x_[1,n_l],\dots,x_[n_k,n_l]))}
collections of effect
vectors of length \eqn{n_{j,l}}{n_jl} }
}

\deqn{\mbox{predictor}(y^1, \ldots, y^p) \sim
\sum_{l=1}^{n_l} A^l \sum_{k=1}^{n_k} g(k, x^{k,l})
= \tilde{A} \sum_{k=1}^{n_k} g(k, \tilde{x}^k) 
}{ predictor(y^1, \ldots, y^p) ~ sum_{l=1}^{n_l} A^l sum_{k=1}^{n_k}
g(k, x^{k,l}) = tilde{A} sum_{k=1}^{n_k} g(k, tilde{x}^k) }
where
\deqn{\tilde{A} = \mbox{cbind}\left( A^1, \ldots, A^{n_l} \right)
}{ tilde{A} = cbind( A^1, ..., A^{n_l} ) }
and
\deqn{\tilde{x}^k = \mbox{rbind}\left( x^{k,1}, \ldots, x^{k,n_l} \right)
}{ tilde{x}^k = rbind( x^{k,1}, ..., x^{k,n_l} ) }
and for each block \eqn{l}{l}, any missing
\eqn{x^{k,l}} is replaced by an \code{NA} vector.
}
\section{Functions}{
\itemize{
\item \code{inla.stack.remove.unused}: Remove unused entries from an existing stack

\item \code{inla.stack.compress}: Compress an existing stack by removing duplicates

\item \code{inla.stack.sum}: Create data stack as a sum of predictors

\item \code{inla.stack.join}: Join two or more data stacks

\item \code{inla.stack.index}: Extract tagged indices

\item \code{inla.stack.LHS}: Extract data associated with the "left hand side" of the model
(e.g. the data itself, \code{Ntrials}, \code{link}, \code{E})

\item \code{inla.stack.RHS}: Extract data associated with the "right hand side" of the model
(all the covariates/predictors)

\item \code{inla.stack.data}: Extract data for an inla call, and optionally join with other variables

\item \code{inla.stack.A}: Extract the "A matrix" for control.predictor
}}

\section{Functions}{
 \itemize{
\item \code{inla.stack.remove.unused}: Remove
unused entries from an existing stack

\item \code{inla.stack.compress}: Compress an existing stack by removing
duplicates

\item \code{inla.stack}: Shorthand for inla.stack.join and inla.stack.sum

\item \code{inla.stack.sum}: Create data stack as a sum of predictors

\item \code{inla.stack.join}: Join two or more data stacks

\item \code{inla.stack.index}: Extract tagged indices

\item \code{inla.stack.LHS}: Extract data associated with the "left hand
side" of the model (e.g. the data itself, \code{Ntrials}, \code{link},
\code{E})

\item \code{inla.stack.RHS}: Extract data associated with the "right hand
side" of the model (all the covariates/predictors)

\item \code{inla.stack.data}: Extract data for an inla call, and optionally
join with other variables

\item \code{inla.stack.A}: Extract the "A matrix" for control.predictor }
}

\examples{

n <- 200
loc <- matrix(runif(n*2), n, 2)
mesh <- inla.mesh.2d(loc.domain = loc,
                     max.edge=c(0.05, 0.2))
proj.obs <- inla.mesh.projector(mesh, loc = loc)
proj.pred <- inla.mesh.projector(mesh, loc = mesh$loc)
spde <- inla.spde2.pcmatern(mesh,
                            prior.range = c(0.01, 0.01),
                            prior.sigma = c(10, 0.01))

covar <- rnorm(n)
field <- inla.qsample(n = 1, Q = inla.spde.precision(spde, theta = log(c(0.5, 1))))[,1]
y <- 2*covar + inla.mesh.project(proj.obs, field)

A.obs <- inla.spde.make.A(mesh, loc = loc)
A.pred = inla.spde.make.A(mesh, loc = proj.pred$loc)
stack.obs <-
    inla.stack(data=list(y=y),
               A=list(A.obs, 1),
               effects=list(c(list(Intercept = 1),
                              inla.spde.make.index("spatial", spde$n.spde)),
                            covar=covar),
               tag="obs")
stack.pred <-
    inla.stack(data=list(y=NA),
               A=list(A.pred),
               effects=list(c(list(Intercept = 1),
                              inla.spde.make.index("spatial", mesh$n))),
               tag="pred")
stack <- inla.stack(stack.obs, stack.pred)

formula <- y ~ -1 + Intercept + covar + f(spatial, model=spde)
result1 <- inla(formula,
                data=inla.stack.data(stack.obs, spde = spde),
                family="gaussian",
                control.predictor = list(A = inla.stack.A(stack.obs),
                                        compute = TRUE))

plot(y, result1$summary.fitted.values[inla.stack.index(stack.obs,"obs")$data, "mean"],
     main = "Observations vs posterior predicted values at the data locations")

result2 <- inla(formula,
                data=inla.stack.data(stack, spde = spde),
                family="gaussian",
                control.predictor = list(A = inla.stack.A(stack),
                                         compute = TRUE))

field.pred <- inla.mesh.project(proj.pred,
  result2$summary.fitted.values[inla.stack.index(stack,"pred")$data, "mean"])
field.pred.sd <- inla.mesh.project(proj.pred,
  result2$summary.fitted.values[inla.stack.index(stack,"pred")$data, "sd"])

plot(field, field.pred, main = "True vs predicted field")
abline(0, 1)
image(inla.mesh.project(mesh,
                        field = field,
                        dims = c(200,200)),
      main = "True field")
image(inla.mesh.project(mesh,
                        field = field.pred,
                        dims = c(200,200)),
      main = "Posterior field mean")
image(inla.mesh.project(mesh,
                        field = field.pred.sd,
                        dims = c(200,200)),
      main = "Prediction standard deviation")
plot(field, (field.pred - field) / 1,
     main = "True field vs standardised prediction residuals")

}
\seealso{
\code{\link[=inla.spde.make.A]{inla.spde.make.A()}}, \code{\link[=inla.spde.make.index]{inla.spde.make.index()}}
}
\keyword{fmesher}
